# vim:ft=perl:cindent:ts=4:sts=4:sw=4:et:fdm=marker:cms=\ #\ %s
#
package Debian::Dwww::ConfigFile;

use Exporter();
use Sys::Hostname;
use strict;

use vars qw(@ISA @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(ReadConfigFile WriteConfigFile);

sub ReadConfigFile($) {
    my $filename = shift;
    my $hostname = &hostname();
    $hostname    =~ s/\..*$//;
    my $cfgvars = {
        #   Basic configuration variables
        'DWWW_SERVERNAME'  => {
            sort   => 10,
            defval => 'localhost',
            descr  => 'Name of the www server dwww(1) will connect to.'
        },
        'DWWW_SERVERPORT'  => {
            sort   => 10,
            defval => 80,
            descr  => 'Port of the www server dwww(1) will connect to.'
        },
        'DWWW_USEHTTPS'  => {
            sort   => 15,
            defval => 'no',
            descr  => 'Should dwww(1) use the HTTPS protocol for connecting.'
        },
        'DWWW_DOCROOTDIR'  => {
            sort   => 10,
            defval => '/var/www',
            descr  => 'The document root for web server.'
        },
        'DWWW_CGIDIR'  => {
            sort   => 10,
            defval => '/usr/lib/cgi-bin',
            descr  => 'The  directory  which contains the CGI scripts for the web server.'
        },
        'DWWW_CGIUSER'  => {
            sort   => 10,
            defval => 'www-data',
            descr  => 'Name of the user the web server uses to execute CGI script.'
        },
        # Browser variables
        'DWWW_BROWSER'  => {
            sort   => 20,
            defval => '',
            descr  => 'Web-browser used by dwww(1) to load dwww main page.'
        },
        'DWWW_X11_BROWSER'  => {
            sort   => 20,
            defval => '',
            descr  => 'Web-browser used by dwww(1) to load dwww main page in X11 mode.'
        },
        #  Path variables
        'DWWW_DOCPATH'  => {
            sort   => 30,
            defval => '/usr/share/doc:/usr/share/info:/usr/share/man:/usr/local/share/doc:'
                       . '/usr/local/share/info:/usr/local/share/man:/usr/share/common-licenses',
            descr  => 'Colon-delimited list of directories dwww-convert(8) supplies files from.'
        },
        'DWWW_ALLOWEDLINKPATH'  => {
            sort   => 35,
            defval => '/usr/share:/usr/lib:/usr/local/share:/var/www',
            descr  => 'Colon-delimited list of directories that could be targets of symlinks.'
        },
        #  Cache files locations
        'DWWW_QUICKFIND_DB'  => {
            sort   => 40,
            defval => '/var/cache/dwww/quickfind.dat',
            descr  => 'Location of cache of installed packages and programs.'
        },
        'DWWW_REGDOCS_DB'  => {
            sort   => 45,
            defval => '/var/cache/dwww/regdocs.dat',
            descr  => 'Location of cache of doc-base registered documents.'
        },
        'DWWW_DOCBASE2PKG_DB'  => {
            sort   => 45,
            defval => '/var/cache/dwww/docbase2pkg.dat',
            descr  => 'Location of cache of doc-base file to package mapping.'
        },
        # Documentation indexing variables
        'DWWW_INDEX_DOCUMENTATION'  => {
            sort   => 50,
            defval => 'yes',
            descr  => 'Should dwww-index++(8) index documentation with swish++.'
        },
        'DWWW_INDEX_FULL_TIME_INTERVAL'  => {
            sort   => 50,
            defval => 28,
            descr  => 'How often (in days) dwww-index++(8) will generate full index of documentation.'
        },
        'DWWW_INDEX_FULL_SLEEP_TIME' => {
            sort   => 50,
            defval => 0.15,
            descr  => 'How long (in seconds) dwww-index++ should sleep after each file in order to not impact regular server operation.'
        },
        'DWWW_INDEX_INCREMENTAL_TIME_INTERVAL'  => {
            sort   => 50,
            defval => 7,
            descr  => 'How often (in days) dwww-index++(8) will generate incremental index of documentation.'
        },
        'DWWW_MERGE_MAN2HTML_INDEX'  => {
            sort   => 55,
            defval => 'no',
            descr  => 'Should dwww-index++(8) use index generated by man2html.'
        },
        # Other variables
        'DWWW_KEEPDAYS'  => {
            sort   => 60,
            defval => 10,
            descr  => 'How many days should dwww-refresh-cache(8) keep documents that have not been accessed.'
        },
        'DWWW_TMPDIR'  => {
            sort   => 60,
            defval => '/var/lib/dwww',
            descr  => 'Temporary directory used by dwww-build(8) and dwww-build-menu(8).'
        },
        'DWWW_USEFILEURL'  => {
            sort   => 60,
            defval => 'no',
            descr  => 'Should dwww-convert(8) generate file:/ urls.'
        },
        'DWWW_TITLE'  => {
            sort   => 60,
            defval => 'dwww: ' . $hostname,
            descr  => 'Title to appear on dwww generated files.'
        },
        'DWWW_USE_CACHE'  => {
            sort   => 60,
            defval => 'yes',
            descr  => 'Should dwww cache accessed documents?'
        }
    };

    if ($filename && -r $filename)
    {
        open DWWWCONF, "<", $filename or die "Can't open $filename: $!\n";
        while (<DWWWCONF>) {
            chomp();
            next if /^\s*#/;
            next if /^\s*$/;
            # Note: the same regexp is used in debian/dwww.config
            if (m/^\s*([^=\s]+)=(['"]?)(.*?)(['"]?)\s*$/ && $2 eq $4) {
                $cfgvars->{$1}->{'cfgval'} = $3;
            } else {
                print STDERR "Warning: unrecognized line in dwww.conf: $_\n";
            }
        }
        close DWWWCONF or die "Can't close $filename: $!\n";
    }

    return $cfgvars;
}

sub WriteConfigFile($$)
{
    my $filename = shift;
    my $cfgvars  = shift;

    open DWWWCONF, ">", "$filename" or die "Can't open $filename for writing: $!\n";
    print DWWWCONF "## Dwww configuration file, see dwww(7) for details.\n";
    print DWWWCONF "## The file is in Bourne shell script format, thus no spaces\n";
    print DWWWCONF "## may be used in variables assignments.\n\n";
    foreach my $key(sort { my $s1 = defined $cfgvars->{$a}->{'sort'} ? $cfgvars->{$a}->{'sort'} : 999;
                           my $s2 = defined $cfgvars->{$b}->{'sort'} ? $cfgvars->{$b}->{'sort'} : 999;
                           my $v  = $s1 <=> $s2;
                           return $v == 0 ? $a cmp $b: $v;
                           }  keys %$cfgvars)
    {
        my $defval = $cfgvars->{$key}->{'defval'};
        my $cfgval = $cfgvars->{$key}->{'cfgval'};
        my $descr  = $cfgvars->{$key}->{'descr'};

        print DWWWCONF "## $descr\n" if $descr;
        print DWWWCONF "## default: $defval\n" if $defval;
        if ($cfgval)
        {
            print DWWWCONF "$key='$cfgval'\n";
        }
        elsif (defined $defval)
        {
            print DWWWCONF "# $key='$defval'\n";
        }
        print DWWWCONF "\n";
    }
    close DWWWCONF or die "Can't close $filename: $!\n";
}
1;
