#ifndef _RHEO_NUMBERING_V1_H
#define _RHEO_NUMBERING_V1_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/numbering_rep_v1.h"
#include "rheolef/smart_pointer.h"

/*Class:numbering
NAME: @code{numbering} - global degree of freedom numbering
@cindex  numbering, global degree of freedom
@cindex  polynomial basis
@clindex numbering
SYNOPSIS:
  @noindent
  The @code{numbering} class defines methods that furnish global
  numbering of degrees of freedom. This numbering depends upon
  the degrees of polynoms on elements and upon the continuity
  requirement at inter-element boundary. For instance the
  "P1" continuous finite element approximation has one degree
  of freedom per vertice of the mesh, while its discontinuous
  counterpart has dim(basis) times the number of elements of the
  mesh, where dim(basis) is the size of the local finite element basis.

AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/

namespace rheolef { 
//<numbering:
class numbering : public smart_pointer<numbering_rep> {
public:

// typedefs:
    typedef numbering_rep                rep;
    typedef smart_pointer<numbering_rep> base;
    typedef size_t                       size_type;

// allocators:

    numbering (std::string name = "");
    numbering (numbering_rep* ptr);

    virtual ~numbering() {}

// accessors:

    std::string name() const;
    virtual
    size_type ndof (
              size_type  mesh_map_dimension,
        const size_type* mesh_n_geo,
        const size_type* mesh_n_element) const;

    virtual
    size_type idof (   
        const size_type*      mesh_n_geo,
        const size_type*      mesh_n_element,
        const geo_element&    K, 
        size_type             i_dof_local) const;

    virtual
    void idof (   
        const size_type*        mesh_n_geo,
        const size_type*        mesh_n_element,
        const geo_element&      K, 
	std::vector<size_type>& i_dof) const;

    virtual bool is_continuous() const;
    virtual bool is_discontinuous() const { return !is_continuous(); }

// i/o:

    void dump(std::ostream& out = std::cerr) const;
};
//>numbering:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
inline
numbering::numbering(std::string name)
 : base (rep::make_ptr(name))
{
}
inline
numbering::numbering (numbering_rep* ptr)
 : base (ptr)
{
}
inline
std::string
numbering::name() const
{
  return data().name();
}
inline
void
numbering::dump(std::ostream& out) const
{
    out << "numbering " << name() << std::endl;
}
inline
numbering::size_type
numbering::ndof (
               size_type  mesh_map_dimension,
         const size_type* mesh_n_geo,
         const size_type* mesh_n_element) const
{
    return data().ndof (mesh_map_dimension, mesh_n_geo, mesh_n_element);
}
inline
numbering::size_type 
numbering::idof (   
         const size_type*      mesh_n_geo,
         const size_type*      mesh_n_element,
         const geo_element&    K, 
         size_type             i_dof_local) const
{
    return data().idof (mesh_n_geo, mesh_n_element, K, i_dof_local);
}
inline
void
numbering::idof (   
        const size_type*        mesh_n_geo,
        const size_type*        mesh_n_element,
        const geo_element&      K, 
	std::vector<size_type>& i_dof) const
{
    return data().idof (mesh_n_geo, mesh_n_element, K, i_dof);
}
inline
bool
numbering::is_continuous () const   
{
    return data().is_continuous();
}

}// namespace rheolef
#endif // _RHEO_NUMBERING_V1_H
