/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

const EXPORTED_SYMBOLS = ["LDAPDirectory"];

const { AddrBookDirectory } = ChromeUtils.import(
  "resource:///modules/AddrBookDirectory.jsm"
);
const { XPCOMUtils } = ChromeUtils.import(
  "resource://gre/modules/XPCOMUtils.jsm"
);

XPCOMUtils.defineLazyModuleGetters(this, {
  FileUtils: "resource://gre/modules/FileUtils.jsm",
  QueryStringToExpression: "resource:///modules/QueryStringToExpression.jsm",
  Services: "resource://gre/modules/Services.jsm",
  MailServices: "resource:///modules/MailServices.jsm",
});

/**
 * @implements {nsIAbLDAPDirectory}
 * @implements {nsIAbDirectory}
 */

class LDAPDirectory extends AddrBookDirectory {
  QueryInterface = ChromeUtils.generateQI([
    "nsIAbLDAPDirectory",
    "nsIAbDirectory",
  ]);

  init(uri) {
    this._uri = uri;

    let searchIndex = uri.indexOf("?");
    this._dirPrefId = uri.substr(
      "moz-abldapdirectory://".length,
      searchIndex == -1 ? undefined : searchIndex
    );

    super.init(uri);
  }

  get readOnly() {
    return true;
  }

  get isRemote() {
    return true;
  }

  get isSecure() {
    return this.lDAPURL.scheme == "ldaps";
  }

  get propertiesChromeURI() {
    return "chrome://messenger/content/addressbook/pref-directory-add.xhtml";
  }

  get dirType() {
    return Ci.nsIAbManager.LDAP_DIRECTORY_TYPE;
  }

  get replicationFileName() {
    return this.getStringValue("filename");
  }

  set replicationFileName(value) {
    this.setStringValue("filename", value);
  }

  get replicationFile() {
    return FileUtils.getFile("ProfD", [this.replicationFileName]);
  }

  get protocolVersion() {
    return this.getStringValue("protocolVersion", "3") == "3"
      ? Ci.nsILDAPConnection.VERSION3
      : Ci.nsILDAPConnection.VERSION2;
  }

  set protocolVersion(value) {
    this.setStringValue(
      "protocolVersion",
      value == Ci.nsILDAPConnection.VERSION3 ? "3" : "2"
    );
  }

  get saslMechanism() {
    return this.getStringValue("auth.saslmech");
  }

  set saslMechanism(value) {
    this.setStringValue("auth.saslmech", value);
  }

  get authDn() {
    return this.getStringValue("auth.dn");
  }

  set authDn(value) {
    this.setStringValue("auth.dn", value);
  }

  get maxHits() {
    return this.getIntValue("maxHits", 100);
  }

  set maxHits(value) {
    this.setIntValue("maxHits", value);
  }

  get attributeMap() {
    let mapSvc = Cc[
      "@mozilla.org/addressbook/ldap-attribute-map-service;1"
    ].createInstance(Ci.nsIAbLDAPAttributeMapService);
    return mapSvc.getMapForPrefBranch(this._dirPrefId);
  }

  get lDAPURL() {
    let uri = this.getStringValue("uri") || `ldap://${this._uri.slice(22)}`;
    return Services.io.newURI(uri).QueryInterface(Ci.nsILDAPURL);
  }

  set lDAPURL(uri) {
    this.setStringValue("uri", uri.spec);
  }

  get childCardCount() {
    return 0;
  }

  get childCards() {
    if (Services.io.offline) {
      return this.replicationDB.childCards;
    }
    return super.childCards;
  }

  /**
   * @see {AddrBookDirectory}
   */
  get cards() {
    return new Map();
  }

  /**
   * @see {AddrBookDirectory}
   */
  get lists() {
    return new Map();
  }

  get replicationDB() {
    this._replicationDB?.cleanUp();
    this._replicationDB = Cc[
      "@mozilla.org/addressbook/directory;1?type=jsaddrbook"
    ].createInstance(Ci.nsIAbDirectory);
    this._replicationDB.init(`jsaddrbook://${this.replicationFileName}`);
    return this._replicationDB;
  }

  getCardFromProperty(property, value, caseSensitive) {
    return null;
  }

  search(queryString, searchString, listener) {
    if (Services.io.offline) {
      this.replicationDB.search(queryString, searchString, listener);
      return;
    }
    this._query = Cc[
      "@mozilla.org/addressbook/ldap-directory-query;1"
    ].createInstance(Ci.nsIAbDirectoryQuery);

    let args = Cc[
      "@mozilla.org/addressbook/directory/query-arguments;1"
    ].createInstance(Ci.nsIAbDirectoryQueryArguments);
    args.expression = QueryStringToExpression.convert(queryString);
    args.querySubDirectories = true;
    args.typeSpecificArg = this.attributeMap;

    this._query.doQuery(this, args, listener, this.maxHits, 0);
  }

  useForAutocomplete(identityKey) {
    // If we're online, then don't allow search during local autocomplete - must
    // use the separate LDAP autocomplete session due to the current interfaces
    let useDirectory = Services.prefs.getBoolPref(
      "ldap_2.autoComplete.useDirectory",
      false
    );
    if (!Services.io.offline || (!useDirectory && !identityKey)) {
      return false;
    }

    let prefName = "";
    if (identityKey) {
      // If we have an identity string, try and find out the required directory
      // server.
      let identity = MailServices.accounts.getIdentity(identityKey);
      if (identity.overrideGlobalPref) {
        prefName = identity.directoryServer;
      }
      if (!prefName && !useDirectory) {
        return false;
      }
    }
    if (!prefName) {
      prefName = Services.prefs.getCharPref(
        "ldap_2.autoComplete.directoryServer"
      );
    }
    if (prefName == this.dirPrefId) {
      return this.replicationFile.exists();
    }

    return false;
  }
}

LDAPDirectory.prototype.classID = Components.ID(
  "{8683e821-f1b0-476d-ac15-07771c79bb11}"
);
