// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use inherent::inherent;

use super::CommonMetricData;

use glean::traits::Counter;

use crate::ipc::{need_ipc, with_ipc_payload};
use crate::private::MetricId;

/// Developer-facing API for recording counter metrics that are acting as
/// external denominators for rate metrics.
///
/// Instances of this class type are automatically generated by the parsers
/// at build time, allowing developers to record values that were previously
/// registered in the metrics.yaml file.
#[derive(Clone)]
pub enum DenominatorMetric {
    Parent {
        /// The metric's ID.
        ///
        /// **TEST-ONLY** - Do not use unless gated with `#[cfg(test)]`.
        id: MetricId,
        inner: glean::private::DenominatorMetric,
    },
    Child(DenominatorMetricIpc),
}
#[derive(Clone, Debug)]
pub struct DenominatorMetricIpc(MetricId);

impl DenominatorMetric {
    /// The constructor used by automatically generated metrics.
    pub fn new(id: MetricId, meta: CommonMetricData, numerators: Vec<CommonMetricData>) -> Self {
        if need_ipc() {
            DenominatorMetric::Child(DenominatorMetricIpc(id))
        } else {
            let inner = glean::private::DenominatorMetric::new(meta, numerators);
            DenominatorMetric::Parent { id, inner }
        }
    }

    #[cfg(test)]
    pub(crate) fn metric_id(&self) -> MetricId {
        match self {
            DenominatorMetric::Parent { id, .. } => *id,
            DenominatorMetric::Child(c) => c.0,
        }
    }

    #[cfg(test)]
    pub(crate) fn child_metric(&self) -> Self {
        match self {
            DenominatorMetric::Parent { id, .. } => {
                DenominatorMetric::Child(DenominatorMetricIpc(*id))
            }
            DenominatorMetric::Child(_) => panic!("Can't get a child metric from a child metric"),
        }
    }
}

#[inherent(pub)]
impl Counter for DenominatorMetric {
    fn add(&self, amount: i32) {
        match self {
            DenominatorMetric::Parent { inner, .. } => {
                Counter::add(&*inner, amount);
            }
            DenominatorMetric::Child(c) => {
                with_ipc_payload(move |payload| {
                    if let Some(v) = payload.denominators.get_mut(&c.0) {
                        *v += amount;
                    } else {
                        payload.denominators.insert(c.0, amount);
                    }
                });
            }
        }
    }

    fn test_get_value<'a, S: Into<Option<&'a str>>>(&self, ping_name: S) -> Option<i32> {
        match self {
            DenominatorMetric::Parent { inner, .. } => inner.test_get_value(ping_name),
            DenominatorMetric::Child(c) => {
                panic!("Cannot get test value for {:?} in non-parent process!", c.0);
            }
        }
    }

    fn test_get_num_recorded_errors<'a, S: Into<Option<&'a str>>>(
        &self,
        error: glean::ErrorType,
        ping_name: S,
    ) -> i32 {
        match self {
            DenominatorMetric::Parent { inner, .. } => {
                inner.test_get_num_recorded_errors(error, ping_name)
            }
            DenominatorMetric::Child(c) => {
                panic!(
                    "Cannot get the number of recorded errors for {:?} in non-parent process!",
                    c.0
                );
            }
        }
    }
}

#[cfg(test)]
mod test {
    use crate::{common_test::*, ipc, metrics};

    #[test]
    fn sets_denominator_value_parent() {
        let _lock = lock_test();

        let metric = &metrics::test_only_ipc::an_external_denominator;
        metric.add(1);

        assert_eq!(1, metric.test_get_value("store1").unwrap());
    }

    #[test]
    fn sets_denominator_value_child() {
        let _lock = lock_test();

        let parent_metric = &metrics::test_only_ipc::an_external_denominator;
        parent_metric.add(3);

        {
            // scope for need_ipc RAII
            let child_metric = parent_metric.child_metric();
            let _raii = ipc::test_set_need_ipc(true);
            let metric_id = child_metric.metric_id();

            child_metric.add(42);

            ipc::with_ipc_payload(move |payload| {
                assert_eq!(
                    42,
                    *payload.denominators.get(&metric_id).unwrap(),
                    "Stored the correct value in the ipc payload"
                );
            });
        }

        assert!(
            false == ipc::need_ipc(),
            "RAII dropped, should not need ipc any more"
        );
        assert!(ipc::replay_from_buf(&ipc::take_buf().unwrap()).is_ok());

        assert!(
            45 == parent_metric.test_get_value("store1").unwrap(),
            "Values from the 'processes' should be summed"
        );
    }
}
