program which;

(*
 *  which  -- display full pathname for executable command
 *  Copyright (C) 2000-2002 Trane Francks
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *  
 *  Author and Maintainer:  Trane Francks <trane@gol.com>
 *
 *  To reach me by snail-mail, contact me at:
 *
 *                          Sento Hills Tsurukawa #305
 *                          1856-4 Kanai-cho, Machida City
 *                          Tokyo 195-0071
 *                          JAPAN
 *
 *  Comments, suggestions, and bug reports should be sent to me, preferably
 *  by e-mail. I'd like to hear from you if you find this program useful.
 *  Constructive criticism is nice; flames will go quietly into /dev/null.
 *)

(*
 *  NAME
 *    which -- display full pathname for executable command
 *
 *  SYNOPSIS
 *    which [options] [command ...]
 *
 *  DESCRIPTION
 *    DOS uses the PATH environment variable to determine which directories
 *    to search when attempting to execute a program. For each command
 *    specified, which searches directories defined in PATH and prints the
 *    name of the first executable file that matches command. When /a or -a
 *    is specified on the command line, which returns all executables
 *    found in PATH that match the command name. If the option is /ad or
 *    /d, which prints 'detail' information, i.e., date and file size.
 *  
 *    When used with the 'WHICHDETAILS' environment variable, the details
 *    options (/ad and /d) act as a toggle, disabling detailed output for
 *    that execution. To cause which to print details by default, simply
 *    'SET WHICHDETAILS=<value>', where <value> is anything you like. I
 *    use '1', as it is a single byte and saves valuable environment space.
 *    Feel free to use whatever you like, though; the only thing that's
 *    needed to make WHICHDETAILS work is for it to show up in the
 *    environment.
 *
 *    Typically, one would not specify the file extension of a command to
 *    search for; doing so limits the search to that specific file name.
 *    In such a limited search, it's possible that which would return the
 *    name of an executable that is farther down the PATH than the actual
 *    file that is being executed. For example, if you have TEST.COM,
 *    TEST.EXE, and TEST.BAT all residing in the current directory, specifying
 *    TEST.BAT on the command line would cause which to report that TEST.BAT
 *    is being executed when, in fact, TEST.COM is the program being run. A
 *    better way is to simply search for TEST unless you really need to
 *    find an executable with a specific extension.
 *  
 *    In keeping with the execution order used by DOS, which displays COM
 *    files first, followed by EXE files, and then BAT files (for searches
 *    that do not have the extension specified). which searches the current
 *    directory before searching the directories defined in PATH. To take
 *    into consideration that which may be used on operating systems other
 *    than DOS -- or with command processors other than COMMAND.COM -- which
 *    checks for a list of program extensions in the PATHEXT environment
 *    variable. To use the PATHEXT environment variable, set the variable with
 *    the extentions you intend to use, each extension separated by a semi-
 *    colon. E.g.:
 *
 *    SET PATHEXT=.COM;.EXE;.BAT;.CMD;.VBS;.VBE;.JS;.JSE;.WSF;.WSH
 *
 *    If PATHEXT is not specified in the environment, which defaults to the
 *    DOS-standard extensions of .COM, .EXE and .BAT. If PATHEXT is specified
 *    in the environment, which searches in the order the extensions are
 *    listed. If you're going to use this variable, be sure you understand
 *    the execution order of the command processor you are using. The example
 *    given above is taken from my own NT system -- with standard command
 *    processor and VB Script, Java Script and Windows Scripting Host
 *    installed. Your mileage may vary, so understand your command processor's
 *    execution order before adding PATHEXT to the environment.
 *
 *  OPTIONS
 *    which accepts the following options:
 *
 *    /? *
 *    --help
 *                 Displays a short summary of how to use the program. When
 *                 this switch is in the first position on the command line,
 *                 which ignores all other command-line entries and displays
 *                 help information
 *
 *    /v *
 *    --version
 *                 Displays version and licensing information. When this
 *                 switch is in the first position on the command line, which
 *                 ignores all other command-line entries and displays
 *                 version information
 *
 *    /a *
 *    -a
 *                 Displays all executable files matching command that are
 *                 found in PATH, rather than just the first match. This
 *                 option cannot be used with other options
 *
 *    /ad *
 *    -ad
 *                 Displays all executable files matching command that are
 *                 found in PATH, rather than just the first match. Includes
 *                 file size and date information for the executable. This
 *                 option cannot be used with other options
 *
 *    /d *
 *    -d
 *                 Displays first executable file matching command that is
 *                 found in PATH. Includes file size and date information for
 *                 the executable. This option cannot be used with other
 *                 options
 *
 *  * NOTE: which checks for the switchar defined in DOS. If you have set a
 *          different switchar than the standard "/" character, which will
 *          honour your preferred setting.
 *
 *  ENVIRONMENT VARIABLES
 *    which uses the following environment variables:
 *
 *    PATH          The list of directories to search for command
 *    PATHEXT       The list of program extensions to check (optional)
 *    WHICHDETAILS  Sets default behaviour to display file details (optional)
 *
 *  DIAGNOSTICS
 *    Possible exit values are:
 *
 *    0    Success:
 *             -- An executable file matching command was found
 *             -- Help or version information was displayed
 *    1    Failure due to the following:
 *             -- A matching executable file could not be found
 *             -- Invalid command-line option
 *
 *  NOTE
 *    This command has no way of knowing about COMMAND.COM internal commands
 *    and, therefore, is unable to help you resolve such conflicts. Under
 *    DOS, COMMAND.COM executes internal commands before searching PATH to
 *    find the executable being called.
 *
 *    If which is searching for multiple files and is able to locate one or
 *    more files, the exit value will be 0 even if other commands are not
 *    found. Only the first position on the command line is tested for
 *    options; all other entries will be treated as filenames.
 *
 *  LIMITATIONS
 *    Technically not a bug (rather, a limitation of a 16-bit, real-mode DOS
 *    app), but which will not work correctly on partitions larger than 2 GB.
 *    So, if you want to use which on non-DOS systems, make sure you're looking
 *    for 8.3 filenames on a < 2GB partition with a PATH that doesn't contain
 *    any spaces or other funky characters. In other words, it's a DOS app. If
 *    you try to use which on, say, Windows Me with a 20-gig partition, you're
 *    outta luck.
 *
 *  AUTHOR
 *    Trane Francks <trane@gol.com>
 *
 *    I prefer being contacted via e-mail; however, you can snail-mail me at:
 *
 *      Trane Francks
 *      Sento Hills Tsurukawa #305
 *      1856-4 Kanai-cho, Machida City
 *      Tokyo 195-0071
 *      JAPAN
 *
 *
 *  DOS Extensions		20 March 2002
 *
 *)

(*
 *  which was originally coded in Emacs 19.34.6 for, and compiled under, BTP7.
 *  which v2.0 was coded with Emacs 20.4.1. which v1.0 has compiled without
 *  errors under the Free Pascal Compiler v1.0 on a Linux system, so it is
 *  quite likely that which will compile under the DOS port of the Free Pascal
 *  Compiler and the resulting executable should work correctly. Your mileage
 *  may vary. which 2.1 was coded in Emacs 20.7.1
 *
 *  which now includes a unit called Country (country.pas), which is used in
 *  determining the operating system's time format, time separator, date
 *  format, date separator and thousands separator. This unit was discovered
 *  in the Pascal SWAG snippets with the SWAG SUPPORT TEAM listed as the
 *  contributor. The unit has been modified slightly by me to expose functions
 *  that were not available in the snippet provided in the SWAG archive. My
 *  heartfelt thanks go to the original, unknown contributor of the unit and to
 *  the SWAG team; I would not have been able to so simply add codepage support
 *  to the program without this snippet.
 *)


uses Crt, Dos, Country;

const
   version = '2.1'; { program version number }

var
   path		  : string;  { directories in PATH environment variable }
   pathExt	  : string;  { extensions in PATHEXT environment variable }
   currentDir	  : string;  { current directory to search }
   foundCurrent	  : boolean; { true if current filename is found }
   cmd2Find	  : string;  { user-defined command to search for }
   userInput	  : string;  { temp variable to contain cmd-line parms }
   index	  : integer; { array counter }
   counter	  : byte;    { position on command line - file to find }
   i		  : byte;    { current file position }
   findAll	  : boolean; { true if -a specified; otherwise, false }
   allExt	  : boolean; { search all extensions? Default = true }
   dirIndex	  : integer; { current position in PATH }
   endOfPath	  : boolean; { true if we've exhausted the path }
   foundFile	  : boolean; { true if we found a match; otherwise, false }
   pathExists	  : boolean; { true if PATH is set; otherwise, false }
   cwd		  : string;  { current working directory on default drive }
   dirString	  : dirStr;  { Turbo Pascal fsplit function built-in vars... }
   nameString	  : nameStr; { ...for directory, name, and extension part... }
   extString	  : extStr;  { ...of fully qualified filename }
   details	  : boolean; { true when /d -d /ad -ad options present }
   whichDetails	  : string;  { holds value of WHICHDETAILS env variable }
   extIndex	  : integer; { holds the current position within pathExt }
   currentExt	  : string;  { holds the current extension }
   endOfPathExt	  : boolean; { true if at end of PATHEXT; otherwise, false }
   thouSeparator  : char;    { decimal separator for file size }
   d8Format	  : word;    { 0:mmddyyyy, 1:ddmmyyyy, 2:yyyymmdd }
   d8Separator	  : char;    { e.g., mm-dd-yyyy or mm/dd/yyyy }
   thymeFormat	  : byte;    { 0:12-hour time, 1:24-hour time }
   thymeSeparator : char;    { e.g., hh:mm:ss or hh.mm.ss }
   switchar	  : char;    { the DOS command-switch character }
   done		  : boolean; { true if we've searched for files in last path }

function getswitchar: char;
(*
 *  gets and returns the DOS switch character
 *)
var
   regs	: registers; { holds packed registers for switchar }
begin { function getswitchar }
   regs.ax := $37 shl 8;
   with regs do
      msdos(regs);
   getswitchar := chr(regs.dx and $00ff)
end; { function getswitchar }

   
procedure displayhelp;
(*
 *  display help information
 *)
begin { procedure displayhelp }
   writeln('which ', version);
   writeln('usage: which [option] [command ...]');
   writeln;
   writeln('options: /?  | --help     display help information');
   writeln('         /v  | --version  display version information');
   writeln('         /a  | -a         display all matching executables');
   writeln('         /ad | -ad        display all matches and details');
   writeln('         /d  | -d         display details');
   writeln;
   writeln('examples: which /?');
   writeln('          which edit');
   writeln('          which /a descent turbo move');
end; { procedure displayhelp }


procedure displayversion;
(*
 *  display version information
 *)
begin { procedure display version }
   writeln('which ', version);
   writeln('Copyright (C) 2000, 2001, 2002 Trane Francks');
   writeln('which comes with NO WARRANTY');
   writeln('to the extent permitted by law.');
   writeln('You may redistribute copies of which');
   writeln('under the terms of the GNU General Public License.');
   writeln('For more information about these matters,');
   writeln('see the file LICENSE.TXT.');
   writeln;
   writeln('Report bugs to "Trane Francks" <trane@gol.com>');
   writeln('Snail-mail address: Trane Francks');
   writeln('                    Sento Hills Tsurukawa #305');
   writeln('                    1856-4 Kanai-cho, Machida City');
   writeln('                    Tokyo 195-0071');
   writeln('                    JAPAN');
end; { procedure display version }


procedure dir2search(var index	    : integer;
			 path	    : string;
		     var currentDir : string;
		     var endOfPath  : boolean);
(*
 *  This procedure is used to parse the PATH setting to determine the current
 *  directory to search in. It returns the position in the path along with
 *  the directory to search. Checks to see whether the current working
 *  directory is the same as specified path; if so, skip it and get the next
 *  directory because we check the cwd first!
 *)
begin { procedure dir2search() }
   repeat
      currentDir := ''; { initialize the string to bugger-all }
      if (index >= length(path)) then
	 exit { we're at or beyond the end of path }
      else
	 while ((index <= length(path)) and (path[index] <> ';')) do
	 begin { parse directory }
	    currentDir := concat(currentDir, path[index]);
	    inc(index);
	 end; { parse directory }
      if (index < length(path)) then
	 inc(index) { to move us past the ; character delimiting PATH };
   until (currentDir <> cwd) and ((currentDir + '\') <> cwd);
end; { procedure dir2search() }


procedure ext2search(var extIndex     : integer;
			 pathExt      : string;
		     var currentExt   : string;
		     var endOfPathExt : boolean);
(*
 *  ext2search parses the PATHEXT string to determine the current program
 *  extension for which to search. It returns the position in pathExt along
 *  with the extension for which to search.
 *)
begin
   currentExt := ''; { initialize the string to bugger-all }
   if (extIndex >= length(pathExt)) then { we're at the end of pathExt }
   begin { exit procedure }
      endOfPathExt := true;
      exit
   end { exit procedure }
   else { parse string for new extension }
      while ((extIndex <= length(pathExt)) and (pathExt[extIndex] <> ';')) do
      begin { parse extension }
	 currentExt := concat(currentExt, pathExt[extIndex]);
	 inc(extIndex);
      end; { parse extension }
   if (extIndex < length(pathExt)) then
      inc(extIndex) { to move us past the ; character delimiting PATHEXT }
   else
      endOfPathExt := true;
end; { ext2search }


procedure findcommand(	  cmd2Find     : string;
			  currentDir   : string;
		      var foundFile    : boolean;
		      var foundCurrent : boolean);
(*
 *  findcommand takes the string cmd2Find and searches the given directory
 *  to see if the file exists.
 *
 *  uses: subprocedure printfilename
 *)
var
   dirInfo : searchRec;
   dirString  : dirStr;  { Turbo Pascal fsplit function built-in vars }
   nameString : nameStr;
   extString  : extStr;
function leadingzero(w : word): string;
(*
 *  prepends a zero to unpacked time values
 *
 *  subfunction of procedure findcommand()
 *)
var
   s : string;

begin { function leadingzero() }
   Str(w:0, s);
   if Length(s) = 1 then
      s := '0' + s;
   leadingzero := s;
end; { function leadingzero() }

procedure printdetails;
(*
 *  prints size, date, time
 *
 *  subprocedure of findcommand()
 *)
var
   dt		 : DateTime;
   size		 : string;
   needsCommas	 : boolean;
   leadingDigits : shortint;
   counter	 : byte;
   
begin { subprocedure printdetails }
   gotoxy(20, wherey);
   str(dirInfo.size:0, size);
   if length(size) <= 3 then
      needsCommas := false
   else
      needsCommas := true;
   leadingDigits := (length(size) mod 3);
   if not needsCommas then
      write('Size: ', size)
   else
   begin { insert commas }
      write('Size: ');
      for counter := 1 to length(size) do
      begin { print filesize }
	 if leadingDigits = 0 then { reset leadingDigits }
	    leadingDigits := 3;
	 write(size[counter]);
	 dec(leadingDigits);
	 if (leadingDigits = 0) then
	 begin { print delimiter }
	    if (counter <> length(size)) then
	       write(thouSeparator);
	    leadingDigits := 3;
	 end; { print delimiter }
      end; { print filesize }
   end; { insert commas }
   write(' bytes');
   gotoxy(50, wherey);
   unpacktime(dirInfo.time, dt);
   case d8Format of
     0 : begin { mmddyyyy }
	    write('Date: ', leadingzero(dt.month), d8Separator,
		  leadingzero(dt.day), d8Separator, leadingzero(dt.year), ' ');
	 end; { mmddyyyy }

     1 : begin { ddmmyyyy }
	    write('Date: ', leadingzero(dt.day), d8Separator,
		  leadingzero(dt.month), d8Separator, leadingzero(dt.year),
		  ' ');
	 end; { ddmmyyyy }

     2 : begin { yyyymmdd }
	    write('Date: ', leadingzero(dt.year), d8Separator,
		  leadingzero(dt.month), d8Separator, leadingzero(dt.day),
		  ' ');
	 end; { yyyymmdd }
   else { can't determine date format, so use international eng with hyphens }
      write('Date: ', leadingzero(dt.year), '-', leadingzero(dt.month), '-',
	      leadingzero(dt.day), ' ');
   end; { case d8Format of }
   case thymeFormat of
     0 : begin { 12-hour format }
	    if (dt.hour > 12) then { adjust hours for PM display }
	       write(dt.hour - 12)
	    else { write it raw, it's an AM time }
	       if (dt.hour = 0) then
		  write('12')
	       else
		  write(dt.hour);
	    write(thymeSeparator, leadingzero(dt.min), thymeSeparator,
		  leadingzero(dt.sec));
	    if (dt.hour >= 12) then
	       writeln('p')
	    else
	       writeln('a');
	 end; { 12-hour format }
     
     1 : begin { 24-hour format }
	    writeln(leadingzero(dt.hour), thymeSeparator, leadingzero(dt.min),
		    thymeSeparator, leadingzero(dt.sec));
	 end; { 24-hour format }
   else { can't determine time format, so print 24-hour time with colons }
      writeln(leadingzero(dt.hour), ':', leadingzero(dt.min), ':',
	      leadingzero(dt.sec));
   end; { case thymeFormat of }
   for counter := 1 to 79 do { print file/details separator line }
      write('-');
   writeln;
end; { subprocedure printdetails }

procedure printfilename;
(*
 *  prints filename of matching command in PATH
 *
 *  subprocedure of findcommand()
 *)
begin { subprocedure printfilename }
   if (currentDir = cwd) then { we're in the user's current working directory }
      writeln('.\', dirInfo.name) { don't print absolute path }
   else if (currentDir = '') then { user specified drive & filename }
   begin { split filename }
      fsplit(cmd2Find, dirString, nameString, extString);
      writeln(dirString, dirInfo.name);
   end { split filename }
   else
      writeln(currentDir, dirInfo.name); { print absolute path }
   if details then
      printdetails;
end; { subprocedure printfilename }



begin { procedure findcommand() }
   if ((currentDir <> '') and (currentDir[length(currentDir)] <> '\')) then
      currentDir := concat(currentDir, '\');
   cmd2Find := concat(currentDir, cmd2Find);
   findfirst(cmd2Find, archive, dirInfo);
   if ((pos('*', cmd2Find) <> 0) or (pos('?', cmd2Find) <> 0)) then
   begin { process wildcards }
      while (dosError = 0) do
      begin { loop through matching file spec }
	 printfilename;
	 findnext(dirInfo);
	 foundFile := true;
	 foundCurrent := true;
      end; { loop through matching file spec }
   end { process wildcards }
   else
      if (dosError = 0) then
      begin { only process single filename }
	 printfilename;
	 foundFile := true;
	 foundCurrent := true;
      end; { only process single filename }
end; { procedure findcommand() }
   

begin { program which }
   { begin initialization }
   directVideo := false; { use BIOS screen-writes to support other codepages }
   findAll := false; { default behaviour is to find first executable only }
   allExt := true; { default behaviour is to check all extensions }
   dirIndex := 1; { start at the beginning of PATH }
   extIndex := 1; { start at the beginning of PATHEXT }
   foundCurrent := false; { we haven't found any files yet... }
   foundFile := false; { we haven't found any files yet... }
   details := false; { default behaviour is to print path/filename only }
   endOfPathExt := false; { we haven't started checking yet }
   done := false;
   switchar := getswitchar; { get command-line switch char from DOS }
   path := getenv('PATH');
   if (path = '') then
      pathExists := false
   else
      pathExists := true;
   pathExt := getenv('PATHEXT');
   if (pathExt = '') then
      pathExt := '.COM;.EXE;.BAT'; { go with default extensions }
   whichDetails := getenv('WHICHDETAILS');
   if (whichDetails <> '') then
      details := true;
   { get codepage-related information }
   d8Format := dateFormat;
   d8Separator := dateSeparator;
   thymeFormat := timeFormat;
   thymeSeparator := timeSeparator;
   thouSeparator := thousandSeparator;
   
   if (paramcount = 0) then
   begin { no parameters given }
      writeln('which: no filename specified');
      displayhelp;
      halt(1);
   end { no parameters given }
   else
   begin { check specified parameters }
      userInput := paramstr(1);
      for index := 1 to length(userInput) do { uppercase user input }
	 userInput[index] := upcase(userInput[index]);
      if ((userInput = '--HELP') or (userInput = switchar + '?')) then
      begin { display help }
	 displayhelp;
	 halt(0);
      end { display help }
      else if ((userInput = '--VERSION') or (userInput = switchar + 'V')) then
      begin { display version }
	 displayversion;
	 halt(0);
      end; { display version }
      if (((userInput = '-A') or (userInput = switchar + 'A'))and
	  (paramcount > 1)) then
	 findAll := true
      else if (((userInput = '-A') or (userInput = switchar + 'A') or
	       (userInput = '-AD') or (userInput = switchar + 'AD') or
	       (userInput = '-D') or (userInput = switchar + 'D')) and
	       (paramcount = 1)) then
      begin { no filenames specified }
	 writeln('which: no filename specified');
	 displayhelp;
	 halt(1);
      end { no filesnames specified }
      else if (((userInput = '-AD') or (userInput = switchar + 'AD')) and
	       (paramcount > 1)) then
      begin { look for all files/print details }
	 findAll := true;
	 if (whichDetails = '') then
	    details := true
	 else
	    details := false;
      end { look for all files/print details }
      else if (((userInput = '-D') or (userInput = switchar + 'D')) and
	       (paramcount > 1)) then
	 if (whichDetails = '') then
	    details := true
	 else
	    details := false
      else if ((userInput[1] = '-') or (userInput[1] = switchar)) then
      begin { invalid option specified }
	 writeln('which: ', paramstr(1), ': invalid option');
	 displayhelp;
	 halt(1);
      end; { invalid option specified }
   end; { check specified parameters }
   begin { find files }
      { First, initialize paramstr(counter) to pull the first filename }
      if (findAll = true) or ((details = true) and (whichDetails = '')) then
	 counter := 2
      else
	 counter := 1;
      getdir(0, cwd); { search current drive/directory first }
      if not (cwd[length(cwd)] = '\') then
         cwd := concat(cwd, '\'); { add trailing backslash if not in root }
      currentDir := cwd;
      for i := counter to paramcount do
      begin { work through all filenames on command line }
	 if (i <= paramcount) then
	    done := false;
	 while not done and ((findAll = true) or
					(foundFile = false) or
					(foundCurrent = false)) do
	 begin {check files }
	    cmd2Find := paramstr(i);
	    fsplit(cmd2Find, dirString, nameString, extString);
	    if (dirString <> '') then
	       { we set currentDir to null so that we don't try to work our }
	       { way through PATH later on; the user is looking for a file }
	       { in a specific location }
	       currentDir := '';
	    for index := 1 to length(cmd2Find) do { uppercase command name }
	       cmd2Find[index] := upcase(cmd2Find[index]);
	    if (extString <> '') then { user specified extension }
	    begin { find fully qualified filename }
	       allExt := false;
	       findcommand(cmd2Find, currentDir, foundFile, foundCurrent);
	    end { find fully qualified filename }
	    else
	       while not endOfPathExt do
	       begin { search through all extensions }
		  if (extIndex > 1) then
		     delete(cmd2Find, length(cmd2Find) -
			    (length(currentExt) - 1), length(currentExt));
		  ext2search(extIndex, pathExt, currentExt, endOfPathExt);
		  cmd2Find := concat(cmd2Find, currentExt);
		  findcommand(cmd2Find, currentDir, foundFile,
			      foundCurrent);
		  if (foundCurrent = true) and (findAll = false) then
		     break;
	       end; { search through all extensions }
	    if ((endOfPath = false) and (pathExists = true) and
		(currentDir <> '')) and ((findAll = true) or
					 (foundCurrent = false)) then
	       {work through path }
	       dir2Search(dirIndex, path, currentDir, endOfPath)
	    else
	       done := true;
	    if (currentDir = '') then { set endOfPath to avoid endless loop }
	    begin
	       endOfPath := true;
	       done := true;
	    end;
	    endOfPathExt := false;
	    extIndex := 1;
	 end; { check files }
	 if (path <> '')  then
	    endOfPath := false;
	 endOfPathExt := false;
	 currentDir := cwd;
	 dirIndex := 1;
	 extIndex := 1;
	 foundCurrent := false;
      end; { work through all filenames on command line }
   end; { find files }
   if (foundFile = false) then
      halt(1)
   else
      halt(0);
end. { program which }
