{
    This file is the source for a series of routines to ease the probing of
    the PCI bus found on most current systems.
    Copyright (C) 1998 by Phil Brutsche

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA  02111-1307  USA.
}

unit pci;

interface

type
  pci_data_block = record
    vendor, device : word;
    command_register : word;
    status_register : word;
    revision : byte;
    programming_interface : byte;
    subclass : byte;
    class : byte;
    cache_line_size : byte;
    latency_timer : byte;
    header_type : byte;
    self_test_result : byte;
    {other_data : array [1..240] of byte;}
    other_data : array [1..44] of byte;
    irq_line : byte;
    irq_pin : byte;
    other_data2 : array [1..194] of byte;
  end;

  ppci_device_info = ^pci_device_info;
  pci_device_info = record
    bus : byte;
    device : byte;
    func : byte;
    data_block_crc : word;
    dev_data : pci_data_block;
  end;
  pci_info_block = record
    signature : array [1..4] of char;
    {pmode_entry_point : longint;}
    pmode_entry_point : record
      segment, offset : word;
    end;
    characteristics : byte;
    major : byte;
    minor : byte;
    last_pci_bus : byte;
  end;


procedure pci_scan_bus (var info_block : pci_info_block);
function pci_resolve_class (class : word) : string;
procedure pci_get_device_info (num : integer; var block : pci_device_info);
function pci_count_devices : integer;
function pci_read_config_byte (bus, device, func : byte; register : word;
                               var value : byte) : byte;
function pci_read_config_word (bus, device, func : byte; register : word;
                               var value : word) : byte;
function pci_read_config_dword (bus, device, func : byte; register : word;
                               var value : longint) : byte;
function pci_detect (var info : pci_info_block) : boolean;
function pci_find_device (vendor, device_id, index : word) : byte;
function pci_get_errorstr (number : byte) : string;
function pci_get_device (bus, device, func : integer; var info : array of byte) : boolean;

implementation

uses crt, strs, crc, dos;

const pci_device_count = 63;
var
  pci_data_array : array [1..pci_device_count] of ppci_device_info;
  pci_number_devices : byte;
  old_exitproc : pointer;

function pci_get_device (bus, device, func : integer; var info : array of byte) : boolean;
var
  data, result : byte;
  i : word;
  ch : char;
  status, vendor, device_id : word;
  revision, device_type : byte;
begin
  pci_get_device := false;
  if pci_read_config_word (bus, device, func, 0, vendor) <> $00 then
    exit;
  if pci_read_config_word (bus, device, func, 2, device_id) <> $00 then
    exit;
  if (vendor <> $FFFF) and (vendor <> $0000) and
     (device_id <> $ffff) and (device_id <> $0000) then begin
    pci_get_device := true;
    for i := 0 to 255 do begin
      result := pci_read_config_byte (bus, device, func, i, data);
      info [i] := data;
      if result <> 0 then begin
        pci_get_device := false;
        exit;
      end;
    end;
  end;
end;

function pci_read_config_word (    bus, device, func : byte;
                                   register : word;
                               var value : word) : byte;
var
  regs : registers;
begin
  regs.ax := $B109;
  regs.bh := bus;
  regs.bl := (device shl 3) or (func and $07);
  regs.di := register;
  intr ($1a, regs);
  if (regs.ah) = 0 then
    value := regs.cx;
  pci_read_config_word := regs.ah;
end;

function pci_read_config_dword (    bus, device, func : byte;
                                    register : word;
                                var value : longint) : byte;
var
  regs : registers;
  low, high : word;
  result, devfn : byte;
begin
  devfn := (device shl 3) or (func and $07);
  asm
    mov ax, $b10a
    mov bh, bus
    mov bl, devfn
    mov di, register
    int $1a
    mov result, ah
    db $66; mov word ptr value, cx
  end;
  pci_read_config_dword := result;
end;

function pci_read_config_byte (    bus, device, func : byte;
                                   register : word;
                               var value : byte) : byte;
var
  regs : registers;
begin
  regs.ax := $B108;
  regs.bh := bus;
  regs.bl := (device shl 3) or (func and $07);
  regs.di := register;
  intr ($1a, regs);
{  if (regs.ah) = 0 then
    value := regs.cl;}
  if ((regs.flags and fcarry) = 0) and (regs.ah = 0) then
    value := regs.cl;
  pci_read_config_byte := regs.ah;
end;

function pci_detect (var info : pci_info_block) : boolean;
label done;
var
  major, minor, last_bus, charistics, result : byte;
  entry_point, sig : longint;
begin
  asm
    mov ax, $b101
    db $66; xor di, di
    int $1a
    mov result, ah
    cmp ah, $00
    jne done
    mov major, bh
    mov minor, bl
    mov last_bus, cl
    mov charistics, al
    db $66; mov word ptr entry_point, di
    db $66; mov word ptr sig, dx
  end;
  info.major := major;
  info.minor := minor;
  info.last_pci_bus := last_bus;
  info.characteristics := charistics;
  move (entry_point, info.pmode_entry_point, 4);
  move (sig, info.signature, 4);
done:
  pci_detect := not boolean (result);
end;

function pci_find_device (vendor, device_id, index : word) : byte;
var
  result : byte;
begin
  asm
    mov ax, $b102
    mov cx, device_id
    mov dx, vendor
    mov si, index
    int $1a
    mov result, ah
  end;
  pci_find_device := result;
end;

function pci_get_errorstr (number : byte) : string;
begin
  case number of
    $00 : pci_get_errorstr := 'successful';
    $81 : pci_get_errorstr := 'unsupported function';
    $83 : pci_get_errorstr := 'bad vendor ID';
    $86 : pci_get_errorstr := 'device not found';
    $87 : pci_get_errorstr := 'bad PCI register number';
  end;
end;


function pci_resolve_class (class : word) : string;
begin
  case class of
    { drive controller }
    $0100 : pci_resolve_class := 'SCSI controller';
    $0101 : pci_resolve_class := 'IDE controller';
    $0102 : pci_resolve_class := 'NEC 765-compatible floppy controller';
    $0103 : pci_resolve_class := 'IPI controller';
    $0104 : pci_resolve_class := 'RAID controller';
    $0180 : pci_resolve_class := 'other';
    { network adaptor }
    $0200 : pci_resolve_class := 'Ethernet';
    $0201 : pci_resolve_class := 'Token Ring';
    $0202 : pci_resolve_class := 'FDDI';
    $0203 : pci_resolve_class := 'ATM';
    $0280 : pci_resolve_class := 'other';
    { video }
    $0300 : pci_resolve_class := 'VGA';
    $0301 : pci_resolve_class := 'SuperVGA';
    $0302 : pci_resolve_class := 'XGA';
    $0380 : pci_resolve_class := 'other';
    { multi-media controller }
    $0400 : pci_resolve_class := 'video';
    $0401 : pci_resolve_class := 'audio';
    $0480 : pci_resolve_class := 'other';
    { memory }
    $0500 : pci_resolve_class := 'RAM';
    $0501 : pci_resolve_class := 'Flash Memory';
    $0580 : pci_resolve_class := 'other';
    { bridge controller }
    $0600 : pci_resolve_class := 'host processor bridge';
    $0601 : pci_resolve_class := 'ISA bridge';
    $0602 : pci_resolve_class := 'EISA bridge';
    $0603 : pci_resolve_class := 'MicroChannel bridge';
    $0604 : pci_resolve_class := 'PCI bridge';
    $0605 : pci_resolve_class := 'PCMCIA bridge';
    $0606 : pci_resolve_class := 'NuBus bridge';
    $0607 : pci_resolve_class := 'CardBus bridge';
    $0680 : pci_resolve_class := 'other';
    { communications device }
    $0700 : pci_resolve_class := 'XT-compatible RS-232';
    $0701 : pci_resolve_class := 'AT-compatible parallel port';
    $0780 : pci_resolve_class := 'other';
    { system peripherals }
    $0800 : pci_resolve_class := '8259-compatible Programmable Interrupt Controller';
    $0801 : pci_resolve_class := '8237-compatible DMA Controller';
    $0802 : pci_resolve_class := '8254-compatible system timer';
    $0803 : pci_resolve_class := 'real-time clock';
    $0880 : pci_resolve_class := 'other';
    { input device }
    $0900 : pci_resolve_class := 'keyboard controller';
    $0901 : pci_resolve_class := 'digitizer/pen';
    $0902 : pci_resolve_class := 'mouse';
    $0980 : pci_resolve_class := 'other';
    { docking station }
    $0A00 : pci_resolve_class := 'docking station';
    $0A80 : pci_resolve_class := 'other';
    { CPU }
    $0B00 : pci_resolve_class := '386-based';
    $0B01 : pci_resolve_class := '486-based';
    $0B02 : pci_resolve_class := 'Pentium-based';
    $0B03 : pci_resolve_class := 'Pentium-Pro (P6)';
    $0B10 : pci_resolve_class := 'DEC Alpha';
    $0B40 : pci_resolve_class := 'coprocessor';
    { Serial Bus controller }
    $0C00 : pci_resolve_class := 'Firewire (IEEE 1394)';
    $0C01 : pci_resolve_class := 'ACCESS.bus';
    $0C02 : pci_resolve_class := 'SSA';
    $0C03 : pci_resolve_class := 'USB';
  end;
end;

function device_already_found (info_block : ppci_device_info) : boolean;
var
  i : byte;
begin
  device_already_found := false;
  if pci_data_array [1] = nil then exit;
  i := 0;
  repeat
    inc (i);
  until (pci_data_array [i]^.data_block_crc = info_block^.data_block_crc) or
        (pci_data_array [i] = nil);
  if pci_data_array [i] = nil then exit;
  if pci_data_array [i]^.data_block_crc = info_block^.data_block_crc then begin
    device_already_found := true;
    exit;
  end;
end;

procedure pci_scan_bus (var info_block : pci_info_block);
var
  data : array [0..255] of byte;
  data_block : ppci_device_info;
  i, bus, device, func : integer;
begin
  for i := 1 to pci_device_count do
    if pci_data_array [i] <> nil then begin
      freemem (pci_data_array [i], sizeof (pci_data_array [i]^));
      pci_data_array [i] := nil;
    end;
  for bus := 0 to info_block.last_pci_bus do begin
    for device := 0 to 31 do begin
      for func := 0 to 7 do begin
        if pci_get_device (bus, device, func, data) = false then begin
          break
        end else begin
          getmem (data_block, sizeof (pci_device_info));
          move (data, data_block^.dev_data, 256);
          data_block^.data_block_crc := crc16 (data_block^.dev_data, 256);
          data_block^.bus := bus;
          data_block^.device := device;
          data_block^.func := func;
          if device_already_found (data_block) then begin
            freemem (data_block, sizeof (pci_device_info));
            continue;
          end;
          pci_number_devices := pci_number_devices + 1;
          pci_data_array [pci_number_devices] := data_block;
        end;
      end;
    end;
  end;
end;

procedure pci_get_device_info (num : integer; var block : pci_device_info);
begin
  if num <= pci_number_devices then
    move (pci_data_array [num]^, block, sizeof (block));
end;

function pci_count_devices : integer;
begin
  pci_count_devices := pci_number_devices;
end;

{$F+}
procedure pci_local_halt;
var i : integer;
begin
  for i := 1 to pci_device_count do
    if pci_data_array [i] <> nil then begin
      freemem (pci_data_array [i], sizeof (pci_data_array [i]^));
      pci_data_array [i] := nil;
    end;
  exitproc := old_exitproc;
end;
{$F-}

var
  i : integer;
begin
  write ('PCI.TPU initializing...');
  for i := 1 to pci_device_count do
    pci_data_array [i] := nil;
  pci_number_devices := 0;
  old_exitproc := exitproc;
  exitproc := @pci_local_halt;
  writeln ('Done');
end.
