"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPatchDetailsFromCliString = exports.getPackageDetailsFromPatchFilename = void 0;
const path_1 = require("./path");
function parseNameAndVersion(s) {
    const parts = s.split("+");
    switch (parts.length) {
        case 1: {
            return { name: parts[0] };
        }
        case 2: {
            const [nameOrScope, versionOrName] = parts;
            if (versionOrName.match(/^\d+/)) {
                return {
                    name: nameOrScope,
                    version: versionOrName,
                };
            }
            return { name: `${nameOrScope}/${versionOrName}` };
        }
        case 3: {
            const [scope, name, version] = parts;
            return { name: `${scope}/${name}`, version };
        }
    }
    return null;
}
function getPackageDetailsFromPatchFilename(patchFilename) {
    const legacyMatch = patchFilename.match(/^([^+=]+?)(:|\+)(\d+\.\d+\.\d+.*?)(\.dev)?\.patch$/);
    if (legacyMatch) {
        const name = legacyMatch[1];
        const version = legacyMatch[3];
        return {
            packageNames: [name],
            pathSpecifier: name,
            humanReadablePathSpecifier: name,
            path: path_1.join("node_modules", name),
            name,
            version,
            isNested: false,
            patchFilename,
            isDevOnly: patchFilename.endsWith(".dev.patch"),
        };
    }
    const parts = patchFilename
        .replace(/(\.dev)?\.patch$/, "")
        .split("++")
        .map(parseNameAndVersion)
        .filter((x) => x !== null);
    if (parts.length === 0) {
        return null;
    }
    const lastPart = parts[parts.length - 1];
    if (!lastPart.version) {
        return null;
    }
    return {
        name: lastPart.name,
        version: lastPart.version,
        path: path_1.join("node_modules", parts.map(({ name }) => name).join("/node_modules/")),
        patchFilename,
        pathSpecifier: parts.map(({ name }) => name).join("/"),
        humanReadablePathSpecifier: parts.map(({ name }) => name).join(" => "),
        isNested: parts.length > 1,
        packageNames: parts.map(({ name }) => name),
        isDevOnly: patchFilename.endsWith(".dev.patch"),
    };
}
exports.getPackageDetailsFromPatchFilename = getPackageDetailsFromPatchFilename;
function getPatchDetailsFromCliString(specifier) {
    const parts = specifier.split("/");
    const packageNames = [];
    let scope = null;
    for (let i = 0; i < parts.length; i++) {
        if (parts[i].startsWith("@")) {
            if (scope) {
                return null;
            }
            scope = parts[i];
        }
        else {
            if (scope) {
                packageNames.push(`${scope}/${parts[i]}`);
                scope = null;
            }
            else {
                packageNames.push(parts[i]);
            }
        }
    }
    const path = path_1.join("node_modules", packageNames.join("/node_modules/"));
    return {
        packageNames,
        path,
        name: packageNames[packageNames.length - 1],
        humanReadablePathSpecifier: packageNames.join(" => "),
        isNested: packageNames.length > 1,
        pathSpecifier: specifier,
    };
}
exports.getPatchDetailsFromCliString = getPatchDetailsFromCliString;
//# sourceMappingURL=data:application/json;base64,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